# === CONFIGURATION ===
# Name of the VPN interface (Fortinet client) and the WireGuard interface
$vpnInterfaceName = "Fortinet"  # Change if different
$wgInterfaceName = "WireGuard"

# === HELPER FUNCTION ===
function Enable-ICS {
    param (
        [string]$SharedInterface,
        [string]$TargetInterface
    )
    $networkListManager = New-Object -ComObject HNetCfg.HNetShare
    $connections = $networkListManager.EnumEveryConnection()

    foreach ($conn in $connections) {
        $props = $networkListManager.NetConnectionProps($conn)
        $config = $networkListManager.INetSharingConfigurationForINetConnection($conn)

        if ($props.Name -eq $SharedInterface) {
            # Enable sharing on the VPN interface
            if (-not $config.SharingEnabled) {
                Write-Host "→ Enabling ICS on '$SharedInterface'"
                $config.EnableSharing(0)  # 0 = outbound
            }
        }

        if ($props.Name -eq $TargetInterface) {
            # Check if sharing is already enabled on the target interface
            if ($config.SharingEnabled) {
                Write-Host "ICS is already enabled on '$TargetInterface'"
            } else {
                Write-Host "ICS does not need to be enabled on '$TargetInterface'"
            }
        }
    }
}

# === VERIFY INTERFACES ===
$netAdapters = Get-NetAdapter
$vpnFound = $netAdapters | Where-Object { $_.InterfaceDescription -like "*$vpnInterfaceName*" }
$wgFound = $netAdapters | Where-Object { $_.InterfaceDescription -like "*$wgInterfaceName*" }

if (-not $vpnFound) {
    Write-Host "Could not find VPN interface '$vpnInterfaceName'"
    exit 1
}
if (-not $wgFound) {
    Write-Host "Could not find WireGuard interface '$wgInterfaceName'"
    exit 1
}

# === APPLY ICS ===
Enable-ICS -SharedInterface $vpnFound.Name -TargetInterface $wgFound.Name
Write-Host "ICS enabled: '$vpnInterfaceName' + '$wgInterfaceName'"
