# Wireguard install

# Define download URL for WireGuard official MSI installer
$wgInstallerUrl = "https://download.wireguard.com/windows-client/wireguard-installer.exe"
$installerPath = "$env:TEMP\wireguard-installer.exe"

# Download the installer
Write-Host "Downloading WireGuard from official site..."
Invoke-WebRequest -Uri $wgInstallerUrl -OutFile $installerPath

# Run the installer silently
Write-Host "Installing WireGuard..."
Start-Process -FilePath $installerPath -ArgumentList "/install /quiet" -Wait

# Wait for the installation to complete
Start-Sleep -Seconds 10

# Set paths
$wgPath = "C:\Program Files\WireGuard"
$configPath = "$wgPath\Configurations"
$serverConfigFile = "$configPath\server.conf"
$outputPath = "C:\vagrant\files\wireguard"

# Create configuration and output folders if it doesn't exist
New-Item -ItemType Directory -Force -Path $configPath
New-Item -ItemType Directory -Force -Path $outputPath

# Generate server key pair
$serverPrivateKey = & "$wgPath\wg.exe" genkey
$serverPublicKey = $serverPrivateKey | & "$wgPath\wg.exe" pubkey

# Generate client key pair (you can extract and use this later)
$clientPrivateKey = & "$wgPath\wg.exe" genkey
$clientPublicKey = $clientPrivateKey | & "$wgPath\wg.exe" pubkey

# Save the keys for later (optional)
$keysOutput = @"
{
    "Server Private Key": "$serverPrivateKey",
    "Server Public Key":  "$serverPublicKey",
    "Client Private Key": "$clientPrivateKey",
    "Client Public Key":  "$clientPublicKey"
}
"@
$keysOutput | Out-File "$outputPath\generated-keys.json"

# Create the WireGuard server configuration
@"
[Interface]
PrivateKey = $serverPrivateKey
Address = 10.10.0.1/24
ListenPort = 51820

[Peer]
PublicKey = $clientPublicKey
AllowedIPs = 10.10.0.2/32
"@ | Out-File $serverConfigFile -Encoding ascii

# Install the tunnel as a Windows service
& "$wgPath\wireguard.exe" /installtunnelservice $serverConfigFile

# Optionally open the UDP port in Windows Firewall
New-NetFirewallRule -DisplayName "WireGuard VPN" -Direction Inbound -Action Allow `
  -Protocol UDP -LocalPort 51820

# Output server info
Write-Host "WireGuard server has been configured and started."
Write-Host "Keys saved to: $outputPath\generated-keys.json"

# Add shortcuts
## Get the desktop path for the current user
$desktopPath = [Environment]::GetFolderPath("Desktop")
## PowerShell
### Create the shortcut path
$shortcutPath = Join-Path $desktopPath "WireGuard.lnk"
### Create the WScript.Shell COM object
$WshShell = New-Object -ComObject WScript.Shell
### Create the shortcut
$shortcut = $WshShell.CreateShortcut($shortcutPath)
### Set the target path to PowerShell
$shortcut.TargetPath = "C:\Program Files\WireGuard\wireguard.exe"
### Save the shortcut
$shortcut.Save()